package de.knoerig.soundFrequencyMapperFFT;

import java.awt.BorderLayout;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.GridLayout;
import java.awt.HeadlessException;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import javax.sound.sampled.AudioFormat;
import javax.sound.sampled.AudioSystem;
import javax.sound.sampled.DataLine;
import javax.sound.sampled.LineUnavailableException;
import javax.sound.sampled.SourceDataLine;
import javax.sound.sampled.TargetDataLine;
import javax.swing.AbstractAction;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JComboBox;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JSpinner;
import javax.swing.JToggleButton;
import javax.swing.SpinnerNumberModel;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import edu.emory.mathcs.jtransforms.fft.DoubleFFT_1D;

public class SoundFrequencyMapperFFT extends JFrame implements Runnable, ChangeListener, ActionListener {

	public SoundFrequencyMapperFFT() throws HeadlessException
	{
		super(progname);
		setSize(400, 250);
		
		JPanel mainPane = new JPanel();
		mainPane.setLayout(new GridLayout(0,2,5,5));
		
		mainPane.add(new JLabel("Mapping:"));
		mainPane.add(mappingSelector = new JComboBox(mappings));
		mappingSelector.addActionListener(this);
		mappingSelector.setSelectedItem(0);
		
		mainPane.add(new JLabel("FFT length:"));
		mainPane.add(fftLengthSelector = new JComboBox(fftLengths));
		fftLengthSelector.setSelectedIndex(fftLengths.length-1);
		
		mainPane.add(new JLabel("Window:"));
		String windowNames[] = new String[windowType.values().length];
		for(int k=0;k<windowType.values().length;k++) windowNames[k]=windowType.values()[k].toString();
		mainPane.add(windowSelector = new JComboBox(windowNames));
		
		mainPane.add(new JLabel("Sampling rate: "));
		mainPane.add(samplingRateSelector = new JComboBox(samplingrates));
		samplingRateSelector.setSelectedIndex(samplingrates.length-1);
		samplingRateSelector.addActionListener(this);
		
		mainPane.add(new JLabel("lower frequency: "));
		mainPane.add(lowerFrequencySpinner = new JSpinner());
		lowerFrequencySpinner.addChangeListener(this);
		
		mainPane.add(new JLabel("upper frequency: "));
		mainPane.add(upperFrequencySpinner = new JSpinner());
		upperFrequencySpinner.addChangeListener(this);
		
		JPanel buttonPane = new JPanel();
		buttonPane.setLayout(new BoxLayout(buttonPane, BoxLayout.LINE_AXIS));
		buttonPane.setBorder(BorderFactory.createEmptyBorder(0, 10, 10, 10));
		buttonPane.add(Box.createHorizontalGlue());
		buttonPane.add(quitButton= new JToggleButton(actionQuit));
		buttonPane.add(Box.createRigidArea(new Dimension(10, 0)));
		buttonPane.add(startButton = new JToggleButton(actionToggleRun));

		Container contentPane = getContentPane();
		contentPane.setLayout(new BorderLayout(10, 10));
		contentPane.add(mainPane, BorderLayout.CENTER);
		contentPane.add(buttonPane, BorderLayout.PAGE_END);
		
		samplingRateChanged();
	}
	
	protected final byte clipToByteRange(double value)
	{
		if(value <= -128.0) return -128;
		else if(value > 127.0) return 127;
		else return (byte) value;
	}
	
	protected final int frequencyToFFTindex(double f,double f_T,int Nfft)
	{
		int index = (int) ((f/f_T)*Nfft);
		if(index < 1) index=1;
		if(index >= (Nfft>>1)) index=(Nfft>>1)-1;
		return index;
	}
	
	protected final void printVector(String name,double vector[])
	{
		System.out.print(name);
		System.out.print("=[");
		for(int k=0;k<vector.length;k++)
		{
			System.out.print(vector[k]);
			if(k<vector.length-1) System.out.print(',');
			else System.out.println("];");
		}
	}
	
	protected final void printVector(String name,byte vector[])
	{
		System.out.print(name);
		System.out.print("=[");
		for(int k=0;k<vector.length;k++)
		{
			System.out.print(vector[k]);
			if(k<vector.length-1) System.out.print(',');
			else System.out.println("];");
		}
	}
	
	protected final void printFFTVector(String name,double vector[])
	{
		System.out.print(name);
		System.out.print("=[");
		if(vector.length > 0)
		{
			System.out.print('(');
			System.out.print(vector[0]);
			System.out.print("+0i),");
		}
		for(int k=2;k<vector.length-1;k+=2)
		{
			System.out.print('(');
			System.out.print(vector[k]);
			System.out.print('+');
			System.out.print(vector[k+1]);
			System.out.print("i),");
		}
		if(vector.length > 1)
		{
			System.out.print('(');
			System.out.print(vector[1]);
			System.out.print('+');
			System.out.print(vector[1]);
			System.out.println("i)];");
		}
	}
	
	@Override
	public void run() {
		int Nfft = getFFTlength();
		mappingType mapping = getMappingType();
		int f_T = getSamplingRate();
		int fmin=getLowerFrequency();
		int fmax=getUpperFrequency();
		int nmin=frequencyToFFTindex(fmin, f_T, Nfft) << 1;
		int nmax=(frequencyToFFTindex(fmax, f_T, Nfft) << 1)+1;
		int Nh = Nfft >> 1;
		AudioFormat audioformat = new AudioFormat(f_T,8,1,true,false);
		
		try{
			DataLine.Info a_outputinfo = new DataLine.Info(SourceDataLine.class,audioformat);
			DataLine.Info a_inputinfo = new DataLine.Info(TargetDataLine.class,audioformat);
			SourceDataLine audio_output = (SourceDataLine) AudioSystem.getLine(a_outputinfo);
			TargetDataLine audio_input = (TargetDataLine) AudioSystem.getLine(a_inputinfo);  
			byte[] inputbuffer = new byte[Nfft];
			byte[] outputbuffer = new byte[Nh];
			double[] window = getWindow(getWindowType(),Nfft);
			double[] fft_in = new double[Nfft];
			double[] fft_out = new double[Nfft];
			double[] y = new double[Nfft];
			DoubleFFT_1D fft = new DoubleFFT_1D(Nfft);
			
			
			audio_output.open(audioformat);
			audio_output.start();
			audio_input.open(audioformat);
			audio_input.start();
			//FloatControl masterGainCtrl = (FloatControl) audio_output.getControl(FloatControl.Type.MASTER_GAIN);
			//masterGainCtrl.setValue(masterGainCtrl.getMaximum())
			int actualBufferOffest=Nh;
			int oldBufferOffset=0;
			while(shouldRun)
			{
				if(audio_input.read(inputbuffer,actualBufferOffest,Nh)==Nh)
				{	
					// copy segment windowed to the FFT input buffer
					for(int k=0;k<Nh;k++)
					{
						fft_in[k]=window[k]*inputbuffer[k+oldBufferOffset]; // left half - old buffer
						fft_in[k+Nh]=window[k+Nh]*inputbuffer[k+actualBufferOffest]; // right half - actual buffer
					}
					
					//perform forward FFT transform
					fft.realForward(fft_in);
					
					//
					// process spectrum 
					int basebandIdx=2;
					switch(mapping)
					{
					case RangeToBaseBand:
						//copy values between nmin and nmax to baseband
						for(int idx=nmin;idx<=nmax;idx++)
						{
							fft_out[basebandIdx++]=fft_in[idx];
						}
						break;
					case BaseBandToRange:
						//copy baseband to range between nmin and nmax 
						for(int idx=nmin;idx<=nmax;idx++)
						{
							fft_out[idx]=fft_in[basebandIdx++];
						}	
						break;
					case Bandpass:
						// copy values between nmin and nmax
						for(int idx=nmin;idx<=nmax;idx++)
						{
							fft_out[idx]=fft_in[idx];
						}	
						break;
					};
					// inverse FFT
					fft.realInverse(fft_out,true);
					
					for(int k=0;k<Nh;k++)
					{
						outputbuffer[k] = clipToByteRange(y[k+oldBufferOffset]+= fft_out[k]);
						y[k+actualBufferOffest] = fft_out[k+Nh];
					}
					
					audio_output.write(outputbuffer,0,Nh);
					oldBufferOffset=actualBufferOffest;
					actualBufferOffest ^= Nh;
				}
				else
				{
					shouldRun=false;
				}
			}
			audio_input.stop();
			audio_input.close();
			audio_output.stop();
			audio_output.close();
			
		}catch(LineUnavailableException e)
		{
			JOptionPane.showMessageDialog(null,"Error: line not available.\n"+e.getLocalizedMessage(),progname,JOptionPane.ERROR_MESSAGE);
		}
		catch(OutOfMemoryError e)
		{
			JOptionPane.showMessageDialog(null,"Error: out of memory.\n"+e.getLocalizedMessage(),progname,JOptionPane.ERROR_MESSAGE);
		}
		startButton.setSelected(false);
		enableControls(true);
	}

	@Override
	public void stateChanged(ChangeEvent arg0) {
		if(arg0.getSource()==upperFrequencySpinner)
		{
			int fmax=getUpperFrequency();
			int fmin=getLowerFrequency();
			if(fmin >= fmax)
			{
				lowerFrequencySpinner.setValue(new Integer(fmax-1));
			}
		}
		else if(arg0.getSource()==lowerFrequencySpinner)
		{
			int fmax=getUpperFrequency();
			int fmin=getLowerFrequency();
			if(fmin >= fmax)
			{
				lowerFrequencySpinner.setValue(new Integer(fmax-1));
			}
		}

	}

	/**
	 * @brief Calculates the sinc function.
	 * @param x Input value
	 * @return \f$sinc(x)=\left\{\begin{array}{lcl}1&:&x=0\\ \frac{\sin\left(\pi x\right)}{\pi x} &:& x\not= 0\end{array}\right.\f$
	 */
	protected final double sinc(double x)
	{
		if(java.lang.Math.abs(x)<1e-10) return 1.0;
		else
		{
			double val=java.lang.Math.PI*x;
			return java.lang.Math.sin(val)/val;
		}
	}
	
	protected double[] getWindow(windowType type,int WindowSize)
	{
		double[] window = new double[WindowSize];
		switch(type)
		{
		case SinQ:
		{
			double dw=java.lang.Math.PI/WindowSize;
			double w=0.5*dw;
			for(int k=0;k<WindowSize;k++)
			{
				double val=java.lang.Math.sin(w);
				window[k]=val*val;
				w+=dw;
			}
		}
		break;
		case Hamming:
		{
			double dw=2.0*java.lang.Math.PI/(WindowSize-1);
			double w=0.0;
			for(int k=0;k<WindowSize;k++)
			{
				window[k]=0.54+0.46*java.lang.Math.cos(w);
				w+=dw;
			}
		}
		break;
		case Hann:
		{
			double dw=2.0*java.lang.Math.PI/(WindowSize-1);
			double w=0.0;
			for(int k=0;k<WindowSize;k++)
			{
				window[k]=0.5*(1.0-java.lang.Math.cos(w));
				w+=dw;
			}			
		}
		break;
		case Gaussian:
		{
			double sigma=0.4;
			double mu=(WindowSize-1)*0.5;
			double faktor=1.0/(sigma*mu);
			for(int k=0;k<WindowSize;k++)
			{
				double val=(k-mu)*faktor;
				val*=val;
				window[k]=java.lang.Math.exp(-0.5*val);
			}
		}
		break;
		case Lanczos:
		{
			double factor = 2.0/(WindowSize-1);
			for(int k=0;k<WindowSize;k++)
			{
				window[k]=sinc(k*factor-1.0);
			}
		}
		break;
		};
		return window;
	}
	
	protected int getSamplingRate()
	{
		String selection = ((String) samplingRateSelector.getSelectedItem());
		return Integer.parseInt(selection);
	}
	
	protected int getFFTlength()
	{
		String selection = ((String) fftLengthSelector.getSelectedItem());
		return Integer.parseInt(selection);
	}
	
	protected mappingType getMappingType() throws IllegalArgumentException 
	{
		String selection = ((String) mappingSelector.getSelectedItem());
		return mappingType.valueOf(selection);
	}
	
	protected windowType getWindowType() throws IllegalArgumentException 
	{
		String selection = ((String) windowSelector.getSelectedItem());
		return windowType.valueOf(selection);
	}
	
	protected int getLowerFrequency()
	{
		return (Integer) lowerFrequencySpinner.getValue();
	}
	
	protected int getUpperFrequency()
	{
		return (Integer) upperFrequencySpinner.getValue();
	}
	
	protected void samplingRateChanged()
	{
		int nyquistRate = getSamplingRate()/2;
		int fmin=getLowerFrequency();
		int fmax=getUpperFrequency();
		
		if(fmax==0) 
		{
			fmax=20000;
			fmin=15000;
		}
		if(fmax > nyquistRate) fmax=nyquistRate;
		if(fmin >= fmax) fmin=fmax-4000;
		if(fmin < 10) fmin=10;
		lowerFrequencySpinner.setModel(new SpinnerNumberModel(fmin,10,fmax,1));
		upperFrequencySpinner.setModel(new SpinnerNumberModel(fmax,10,nyquistRate,1));
	}
	
	private AbstractAction actionQuit = new AbstractAction("Quit")
	{
		public void actionPerformed(ActionEvent e)
		{
			stopThread();
			dispose();
		}
	};
	
	private AbstractAction actionToggleRun = new AbstractAction("Start")
	{
		public void actionPerformed(ActionEvent e)
		{
			if(startButton.isSelected())
			{
				startThread();
			}
			else
			{
				stopThread();
			}
		}
	};
	
	@Override
	public void actionPerformed(ActionEvent e) 
	{
		if(e.getSource()==samplingRateSelector) samplingRateChanged();
		else if(e.getSource()==mappingSelector)
		{
			switch(getMappingType())
			{
			case RangeToBaseBand:
				mappingSelector.setToolTipText("Maps given frequency range to base band.");
				break;
			case BaseBandToRange:
				mappingSelector.setToolTipText("Maps baseband to given frequency range.");
				break;
			case Bandpass:
				mappingSelector.setToolTipText("Copies only given frequency range (bandpass).");
				break;
			}
		}
	}
	
	/**
	 * @brief Enables or disables the setting controls.
	 * @param enable Set to true to enable the setting controls and to false to disable.
	 */
	private final void enableControls(boolean enable)
	{
		samplingRateSelector.setEnabled(enable);
		lowerFrequencySpinner.setEnabled(enable);
		upperFrequencySpinner.setEnabled(enable);
		fftLengthSelector.setEnabled(enable);
		mappingSelector.setEnabled(enable);
		windowSelector.setEnabled(enable);
	}
	
	private final void stopThread()
	{
		shouldRun = false;
		if(worker != null && worker.isAlive())
		{
			try{
				worker.join();
			}
			catch(InterruptedException ie)
			{
				JOptionPane.showMessageDialog(null,ie.getLocalizedMessage(),progname,JOptionPane.ERROR_MESSAGE);
			}
		}
	}
	
	private final void startThread()
	{
		stopThread();
		enableControls(false);
		shouldRun = true;
		worker = new Thread(this);
		worker.start();
	}
	/**
	 * @brief Program's main function.
	 * @param args Command line arguments.
	 */
	public static void main(String[] args) {
		SoundFrequencyMapperFFT program = new SoundFrequencyMapperFFT();
		program.setVisible(true);
	}
	
	private static final long serialVersionUID = 1L;
	
	private static final String progname="SoundFrequencyMapperFFT";
	
	private static final String samplingrates[]={"8000","22050","44100","96000"};
	
	private enum mappingType { RangeToBaseBand, BaseBandToRange,Bandpass}; 
	
	private static final String mappings[]={ mappingType.RangeToBaseBand.toString(), mappingType.BaseBandToRange.toString(), mappingType.Bandpass.toString()};

	private static final String fftLengths[]={"1024","2048","4096","10240"};
	
	/** window function types */
	private enum windowType { SinQ,Hamming,Hann,Gaussian,Lanczos};
	
	/** window function selections */
	//private static final String windowNames[]={windowType.SinQ.toString(),windowType.Hamming.toString(),windowType.Hann.toString()};
	
	/** toggle button for starting / stopping the sinus generator */
	private JToggleButton startButton;
	
	/** toggle button for quitting the application */
	private JToggleButton quitButton;
	
	/** combo box to select the window function */
	private JComboBox windowSelector;
	
	/** combo box to select the desired mapping */
	private JComboBox mappingSelector;
	
	/** combo box to choose the FFT length */
	private JComboBox fftLengthSelector;
	
	/** combo box to select the sampling rate */
	private JComboBox samplingRateSelector;
	
	/** spinner for selecting the lower frequency bound */
	private JSpinner lowerFrequencySpinner;

	/** spinner for selecting the lower frequency bound */
	private JSpinner upperFrequencySpinner;

	/** thread for performing the audio work */
	private Thread worker = null;
	
	/** flag indicating that the thread should run */
	private boolean shouldRun = true;
}
